import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Calendar, PenTool, CheckCircle, ArrowRight, Paintbrush, Pencil, Layers, Monitor, Quote } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedStyle, setSelectedStyle] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)
  const [formProgress, setFormProgress] = useState(0)

  const styles = [
    { 
      id: 'watercolor', 
      name: 'Акварель', 
      icon: Paintbrush,
      color: '#2C3E50',
      description: 'Легкость и прозрачность красок. Идеально для пейзажей и портретов.',
      duration: '2 часа',
      price: 'от 1200₽',
      quote: 'Акварель учит нас видеть красоту в простоте и прозрачности.',
      features: ['Техника мокрым по мокрому', 'Работа с цветом', 'Создание атмосферы']
    },
    { 
      id: 'graphics', 
      name: 'Графика', 
      icon: Pencil,
      color: '#2C3E50',
      description: 'Точность линий и штриховка. Классическая техника рисунка.',
      duration: '2.5 часа',
      price: 'от 1000₽',
      quote: 'В графике каждая линия имеет значение, каждый штрих — характер.',
      features: ['Работа с карандашом', 'Штриховка и тонировка', 'Композиция']
    },
    { 
      id: 'oil', 
      name: 'Масло', 
      icon: Layers,
      color: '#2C3E50',
      description: 'Богатство цвета и фактуры. Профессиональная живопись.',
      duration: '3 часа',
      price: 'от 1500₽',
      quote: 'Масляная живопись — это диалог между художником и холстом.',
      features: ['Техника мазка', 'Работа с палитрой', 'Создание объема']
    },
    { 
      id: 'digital', 
      name: 'Цифровое', 
      icon: Monitor,
      color: '#2C3E50',
      description: 'Современные технологии в искусстве. Графические планшеты и программы.',
      duration: '2.5 часа',
      price: 'от 1300₽',
      quote: 'Цифровое искусство открывает безграничные возможности творчества.',
      features: ['Работа в Photoshop', 'Графические планшеты', 'Цифровая живопись']
    }
  ]

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (!selectedStyle || !selectedDate) return

    const saved = localStorage.getItem('drawingLessons')
    const lessons = saved ? JSON.parse(saved) : []
    
    const styleData = styles.find(s => s.id === selectedStyle)
    
    const newLesson = {
      id: Date.now(),
      technique: styleData.name,
      instructor: styleData.instructor || 'Преподаватель',
      date: selectedDate,
      rating: 0,
      workTitle: `Работа в технике ${styleData.name.toLowerCase()}`
    }
    
    lessons.push(newLesson)
    localStorage.setItem('drawingLessons', JSON.stringify(lessons))
    
    setIsSubmitted(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="enroll">
      <div className="container">
        <motion.section 
          className="enroll-hero-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="hero-content-enroll-elegant">
            <h1 className="hero-title-enroll-elegant">Записаться на занятие</h1>
            <div className="hero-divider-book"></div>
            <p className="hero-description-enroll-elegant">
              Выберите стиль рисования и дату начала. Каждое занятие — это возможность 
              раскрыть свой творческий потенциал и научиться новым техникам.
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="enroll-form-elegant"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="form-progress-elegant">
            <div className="progress-bar-elegant">
              <motion.div 
                className="progress-fill-elegant"
                initial={{ width: 0 }}
                animate={{ width: `${formProgress}%` }}
                transition={{ duration: 0.5 }}
              />
            </div>
            <div className="progress-text-elegant">{formProgress}%</div>
          </div>

          <form className="enroll-form-wrapper-elegant" onSubmit={handleSubmit}>
            <div className="form-section-style-elegant">
              <div className="form-label-elegant">
                <PenTool size={24} strokeWidth={1.5} />
                <span className="label-text-elegant">Выберите стиль рисования</span>
              </div>
              <div className="styles-elegant-list">
                {styles.map((style, index) => {
                  const Icon = style.icon
                  const isSelected = selectedStyle === style.id
                  
                  return (
                    <motion.button
                      key={style.id}
                      type="button"
                      className={`style-card-elegant ${isSelected ? 'selected' : ''}`}
                      onClick={() => {
                        setSelectedStyle(style.id)
                        setFormProgress(50)
                      }}
                      initial={{ opacity: 0 }}
                      animate={{ opacity: 1 }}
                      transition={{ delay: index * 0.05 }}
                      whileHover={{ y: -2 }}
                      whileTap={{ scale: 0.99 }}
                    >
                      <div className="style-icon-wrapper-elegant">
                        <div className="style-icon-elegant">
                          <Icon 
                            size={40} 
                            strokeWidth={1.5}
                            style={{ color: isSelected ? 'white' : style.color }}
                          />
                        </div>
                        {isSelected && (
                          <div className="style-check-elegant">
                            <CheckCircle size={28} strokeWidth={1.5} />
                          </div>
                        )}
                      </div>
                      <div className="style-content-elegant">
                        <h3 className="style-name-elegant">{style.name}</h3>
                        <p className="style-description-elegant">{style.description}</p>
                        <div className="style-quote-elegant">
                          <Quote size={16} strokeWidth={1.5} />
                          <span>{style.quote}</span>
                        </div>
                        <div className="style-features-elegant">
                          {style.features.map((feature, i) => (
                            <span key={i} className="feature-tag-elegant">
                              {feature}
                            </span>
                          ))}
                        </div>
                        <div className="style-meta-elegant">
                          <div className="meta-item-elegant">
                            <Calendar size={14} strokeWidth={1.5} />
                            <span>{style.duration}</span>
                          </div>
                          <div className="meta-item-elegant">
                            <span>{style.price}</span>
                          </div>
                        </div>
                      </div>
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <AnimatePresence>
              {selectedStyle && (
                <motion.div 
                  className="form-section-date-elegant"
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  exit={{ opacity: 0, height: 0 }}
                  transition={{ duration: 0.4 }}
                  onAnimationComplete={() => setFormProgress(100)}
                >
                  <div className="date-label-elegant">
                    <Calendar size={24} strokeWidth={1.5} />
                    <div className="date-label-content-elegant">
                      <span className="date-label-title-elegant">Дата начала занятий</span>
                      <span className="date-label-hint-elegant">Выберите удобную дату из доступных</span>
                    </div>
                  </div>
                  <input
                    type="date"
                    value={selectedDate}
                    onChange={(e) => {
                      setSelectedDate(e.target.value)
                      setFormProgress(100)
                    }}
                    min={minDateStr}
                    max={maxDateStr}
                    className="date-input-elegant"
                    required
                  />
                </motion.div>
              )}
            </AnimatePresence>

            <motion.button 
              type="submit" 
              className="submit-button-elegant"
              disabled={!selectedStyle || !selectedDate}
              whileHover={{ scale: 1.01 }}
              whileTap={{ scale: 0.99 }}
            >
              <span>Записаться на занятие</span>
              <ArrowRight size={20} strokeWidth={1.5} />
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message-elegant"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={36} strokeWidth={1.5} />
                <span>Вы успешно записаны на занятие!</span>
              </motion.div>
            )}
          </form>
        </motion.section>

        <ImageSlider />

        <motion.section 
          className="instructors-elegant"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="section-header-elegant">
            <h2 className="section-title-elegant">Наши преподаватели</h2>
            <div className="section-divider"></div>
          </div>
          <div className="instructors-elegant-list">
            {[
              { name: 'Мария Иванова', specialty: 'Акварель и графика', experience: '8 лет', rating: '5.0', students: '500+' },
              { name: 'Алексей Петров', specialty: 'Графический рисунок', experience: '10 лет', rating: '5.0', students: '700+' },
              { name: 'Елена Соколова', specialty: 'Масляная живопись', experience: '12 лет', rating: '5.0', students: '600+' },
              { name: 'Дмитрий Новиков', specialty: 'Цифровое искусство', experience: '7 лет', rating: '5.0', students: '400+' }
            ].map((instructor, index) => (
              <motion.div
                key={index}
                className="instructor-elegant-item"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ delay: index * 0.05 }}
                whileHover={{ y: -3 }}
              >
                <div className="instructor-avatar-elegant">
                  <PenTool size={32} strokeWidth={1.5} />
                </div>
                <div className="instructor-content-elegant">
                  <h3 className="instructor-name-elegant">{instructor.name}</h3>
                  <div className="instructor-specialty-elegant">{instructor.specialty}</div>
                  <div className="instructor-stats-elegant">
                    <div className="instructor-stat-elegant">
                      <span className="stat-value-elegant">{instructor.experience}</span>
                      <span className="stat-label-elegant">Опыт</span>
                    </div>
                    <div className="instructor-stat-elegant">
                      <span className="stat-value-elegant">{instructor.rating}</span>
                      <span className="stat-label-elegant">Рейтинг</span>
                    </div>
                    <div className="instructor-stat-elegant">
                      <span className="stat-value-elegant">{instructor.students}</span>
                      <span className="stat-label-elegant">Учеников</span>
                    </div>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

      </div>
    </div>
  )
}

export default Enroll

